# Allplan imports
import NemAll_Python_Geometry as Geometry
import NemAll_Python_BaseElements as BaseElements
import NemAll_Python_IFW_Input as Input

from HandleDirection import HandleDirection
from HandleProperties import HandleProperties

# Local Imports
from ..StateBehaviour import StateBehaviour

from ...Utilities.MouseMessageHelper import MouseMessageType
from ...Utilities.StateBehaviourType import StateBehaviourType
from ... import Preview



class HandleState(StateBehaviour):
    def __init__(self, interactor) -> None:
        super().__init__(interactor)
        self.HandleService: Input.HandleService = interactor.HandleService
        self.StartWithCut = False
        self.lastSelectedHandle = -1
        self.Mirror = False
        self.RailVector: Geometry.Vector3D
        self.RailReference: Geometry.Point3D
        self.RedrawLambda = lambda s, e: self.draw_preview(True)
        self.modelElements = []

    ########################
    # Interactor functions #
    ########################
    def draw_preview(self, Force = False) -> None:
        # Redraw preview only if needed
        #if len(self.modelElements) == 0:   
        if Force:
            self.modelElements = self.interactor.create(self.StartPoint, self.Direction, 1, self.StartWithCut, self.Mirror, True)
        try:
            # Draw elements and handles
            if not self.interactor.InErrorState and len(self.modelElements) > 0:
                BaseElements.DrawElementPreview(self.Document, Geometry.Matrix3D(), self.modelElements, True, [True])
                self.HandleService.DrawHandles()
        except:
            self.modelElements = []
            self.HandleService.RemoveHandles()
            self.interactor.InErrorState = True

    def on_mouse_leave(self) -> None:
        self.draw_preview()
        # Handle is deselected
        # Because when the cursor is outside of the canvas, we do not get the process_mouse_msg events
        # Hence we cannot detect if the MOUSE_UP event was called (handle deselect)
        # This way we fix this bug
        self.SelectedHandle = -1

    def process_mouse_msg(self, MouseMessage, Point, MessageInfo):
        # Get the index of the selected handle (-1 if None)
        index, matrix = self.HandleService.SelectHandle(Point, self.CoordinateInput.GetViewWorldProjection())
        handleModified = False
        if MouseMessage == MouseMessageType.MOUSE_DOWN:
            # If mouse down select the handle
            self.SelectedHandle = index
            self.lastSelectedHandle = self.SelectedHandle
            # If no handle was selected -> the user clicked outside -> finish selection
            if index == -1:
                self.__finish_selection()
            elif self.HandleProps[self.SelectedHandle].handle_id == "Mirror":
                self.Mirror = not self.Mirror
                self.interactor.BuildingElement.Mirror.value = self.Mirror
                handleModified = True
                
        if MouseMessage == MouseMessageType.MOUSE_UP:
            # On mouse up, the handle is deselected
            self.SelectedHandle = -1
            
        if MouseMessage == MouseMessageType.MOUSE_MOVE and self.SelectedHandle != -1:
            # If he mouse event is MOVE, and we have a selected handle
            # Get the input and project on the rail
            inputPoint = self.CoordinateInput.GetInputPoint(MouseMessage, Point, MessageInfo).GetPoint()
            transformed = self.__project(inputPoint)
            #handleModified = False

            if self.HandleProps[self.SelectedHandle].handle_id == "Start":
                # If the start handle is selected, just move the 
                # start point along with all the elements
                self.StartPoint = transformed
                handleModified = True
            elif self.HandleProps[self.SelectedHandle].handle_id == "End":
                # If the end handle is selected, modify the product count
                # The handle 'snaps' to the end of the last product 
                product_count = self.interactor.get_product_count(Geometry.Vector3D(self.StartPoint, transformed))
                direction = Geometry.Vector3D(self.StartPoint, transformed)
                direction.Normalize(product_count * self.interactor.get_product().Length)
                
                # This checks if the direction vector has the same orientation as the original vector
                # DotProduct => |a|*|b|*cos(angleBetween)
                # The only way it is negative is if the cos is -1 -> wrong direction
                dotProduct = self.RailVector.DotProduct(direction)
                if dotProduct >= 0:
                    self.Direction = direction
                    handleModified = True
               
        if MouseMessage == MouseMessageType.DOUBLE_CLICK_LEFT:
            if self.lastSelectedHandle != -1 and self.HandleProps[self.lastSelectedHandle].handle_id != "Mirror":
                self.StartWithCut = not self.StartWithCut
                handleModified = True
        
        if handleModified:
                # If any handle has been modified
                self.HandleService.RemoveHandles()
                self.__create_direction_handles()
                self.HandleService.AddHandles(self.Document, self.HandleProps, Geometry.Matrix3D(), None)
                #self.modelElements = self.interactor.create(self.StartPoint, self.Direction, 1, self.StartWithCut, self.Mirror, True)

        self.draw_preview(handleModified)
        return True

    def on_cancel_function(self) -> bool:
        # Finish selection
        # self.__finish_selection()
        # return False

        # This is temporary until there is a safe alterantive to return False in on_cancel_function()
        return True

    ###################
    # State behaviour #
    ###################
    def enter(self, previous: StateBehaviourType) -> None:
        # Initialize state
        # Set members to default values
        self.HandleProps = []
        self.SelectedHandle = -1
        self.StartWithCut = False

        # Copy start point
        self.StartPoint = Geometry.Point3D(self.interactor.StartPoint) 
        # Modify the direction to be aligned to the end
        product_count = self.interactor.get_product_count(self.interactor.Direction)
        direction = Geometry.Vector3D(self.interactor.Direction)
        direction.Normalize(product_count * self.interactor.get_product().Length)
        self.Direction = direction

        # The rail is the line along which the object can be placed
        # Defined by a reference point and vector
        self.RailReference = Geometry.Point3D(self.StartPoint)
        self.RailVector = Geometry.Vector3D(self.interactor.Direction)

        self.interactor.Palette.InputChanged += self.RedrawLambda

        # Initialize inputs
        self.__create_direction_handles()
        self.HandleService.AddHandles(self.Document, self.HandleProps, Geometry.Matrix3D(), None)
        self.CoordinateInput.InitValueInput(Input.InputStringConvert(self.interactor.Localization.CurrentLanguage["Interactor_ChooseDirection"]), Input.ValueInputControlData())
        self.draw_preview(True)

    def exit(self, next: StateBehaviourType):
        self.interactor.Palette.InputChanged -= self.RedrawLambda
        self.HandleService.RemoveHandles()

    ###########
    # Helpers #
    ###########
    def __finish_selection(self) -> None:
        # Called when the selection is finished
        # Create the model elements
        modelElements = self.interactor.create(self.StartPoint, self.Direction, startWithCut=self.StartWithCut)

        # Create the elements
        BaseElements.CreateElements(self.Document, Geometry.Matrix3D(), modelElements, [], None)

        # Upload data
        self.interactor.upload_data("Create", self.interactor.get_product_count(self.Direction))

        # Tidy up
        self.interactor.HandleService.RemoveHandles()
        self.interactor.change_state(StateBehaviourType.INPUT_START)        

        # Save preview
        maxLevelOfDetail = max(self.interactor.Geometries)
        Preview.Save(self.interactor.get_product().Geometries[maxLevelOfDetail])
    
    def __create_direction_handles(self) -> None:
        # Create the handles for start and end
        self.HandleProps.clear()
        handle = HandleProperties("Start", self.StartPoint, self.StartPoint + self.Direction, [], HandleDirection.VECTOR_DIR)
        handle.handle_type  = Input.ElementHandleType.HANDLE_SQUARE_RED
        handle.info_text    = "Start"
        self.HandleProps.append(handle)

        handle = HandleProperties("End", self.StartPoint + self.Direction, self.StartPoint, [], HandleDirection.VECTOR_DIR)
        handle.handle_type  = Input.ElementHandleType.HANDLE_CIRCLE
        handle.info_text    = "End"
        self.HandleProps.append(handle)

        handle = HandleProperties("Mirror", (self.StartPoint + self.Direction + self.StartPoint) / 2, Geometry.Point3D(), [], HandleDirection.CLICK)
        handle.handle_type  = Input.ElementHandleType.HANDLE_ARROW
        handle.info_text    = "Mirror"
        rot_direction = Geometry.Vector2D(self.Direction.X, self.Direction.Y if self.Direction.Y != 0 else self.Direction.Z)        
        rot_angle = Geometry.Angle()
        rot_angle.SetDeg(90)
        handle.handle_angle = rot_angle + rot_direction.GetAngle() 
        self.HandleProps.append(handle)
        
    def __project(self, point: Geometry.Point3D)-> Geometry.Point3D:
        # Project an input point onto the rail
        point2origin = point - self.RailReference
        projected = self.RailVector.Project(Geometry.Vector3D(point2origin))
        return self.RailReference + projected
