# Compiled Allplan Packages
import NemAll_Python_BaseElements as BaseElements
import NemAll_Python_IFW_ElementAdapter as ElementAdapter
import NemAll_Python_AllplanSettings as Settings
import NemAll_Python_Utility as Utility

# Project packages
from .SchoeckInteractorBase import SchoeckInteractorBase
from ..Style import Style
from ..InputOptions import InputOptions
from ..AttributeQuery import AttributeQuery

# Typing
from typing import Dict, Optional

# Python 
import os
import shutil
import json
import time

# clr and related packages

import clr
clr.AddReference("System")
import System
import System.Collections.Generic as Generic

from .. import PaletteWrapper

from ..Utilities.Languages import get_section, get_culture

class SchoeckUIBase(SchoeckInteractorBase):
    """
    Definition of SchoeckUIBase
    This class implements functions for handling the SchoeckUI
    Root for the PaletteWrapper and the SchoeckUI namespace
    """
    def __init__(self, coordinateInput, pypPath, stringTable, buildingElements, buildingElementComposite, controlProps, modifyUUIDList):
        super().__init__(coordinateInput, pypPath, stringTable, buildingElements, buildingElementComposite, controlProps, modifyUUIDList)
        
        # Initialize SchoeckUI.dll, and link to it
        # It is important to first create the dll and then add reference and load it using clr
        # Create calls an internal Allplan function that has the permissions to load the dll from the Prg folder
        # A normal user will not have these permissions
        self.PaletteWrapper = PaletteWrapper.PaletteWrapper(self.Document, "SchoeckUI.Views.PaletteView", "SchoeckUI.dll")
        clr.AddReference(f"{Settings.AllplanPaths.GetPrgPath()}\\SchoeckUI.dll")
        import SchoeckUI 
        self.SchoeckUI = SchoeckUI

        # Set paths
        self.EnterError = lambda e: self.enter_error_state(e)
        self.SchoeckUI.Globals.ErrorEncountered += self.EnterError

        # Set paths for cache and settings
        self.SchoeckUI.Globals.FileCache.StorageDirectory = os.path.join(Settings.AllplanPaths.GetUsrPath(), "Schoeck\\Cache\\")
        self.SchoeckUI.Globals.ProjectCache.StorageDirectory = os.path.join(Settings.AllplanPaths.GetCurPrjPath(), "Schoeck\\Cache\\")

        settingsPath = os.path.join(Settings.AllplanPaths.GetUsrPath(), "Schoeck\\Settings.xml")
        if not os.path.exists(settingsPath):
            shutil.copyfile(self.SchoeckUI.Globals.Settings.DefaultSettingsPath, settingsPath)

        self.SchoeckUI.Globals.Settings.LoadSettings(settingsPath)
        self.SchoeckUI.Globals.CheckCentralCache()

        # Link the attribute query
        self.SchoeckUI.Globals.AttributeQuery = AttributeQuery(self.Document)

        # Shortcuts to settings and localization and Palette
        self.Settings = self.SchoeckUI.Globals.Settings
        self.Localization = self.SchoeckUI.Globals.Localization
        self.Palette = self.SchoeckUI.Globals.Palette

        # Check if the versions are compatible
        self.SchoeckUI.Globals.AllplanVersion = Settings.AllplanVersion.MainReleaseName()
        if self.SchoeckUI.Globals.VersionValidity == -2:
            Utility.ShowMessageBox(self.Localization.CurrentLanguage["MessageBox_InvalidVersionUpdateNeeded"], Utility.MB_OK)
            self.PaletteWrapper.Close()
            raise RuntimeError("Plugin version is not valid, pluin has to be updated")

        self.__try_set_default_section()
        self.__try_set_default_culture()
        self.__try_set_catalogue_selection()
        
        # Show the palette and make shortcuts to the DataContext
        self.PaletteWrapper.Show(self.get_palette_title())
        self.Data = self.Palette.DataContext

        self.__attributeData: Optional[dict] = None

    def get_palette_title(self) -> str:
        """
        Abstract function that returns the title of the palette
        The title gets set only once when the Palette gets shwon
        """
        raise NotImplementedError("get_palette_title not implemented!")

    def on_cancel_function(self) -> bool:
        """
        Check for input function cancel in case of ESC
        Returns:
            True/False for success.
        """
        # Close the palette
        shouldClose = super().on_cancel_function()
        if shouldClose:
            self.SchoeckUI.Globals.ErrorEncountered -= self.EnterError
            self.PaletteWrapper.Close()
        return shouldClose 

    ##########################
    # UI getters and setters #
    ##########################

    def get_styles(self) -> Dict[str, Style]:
        """
        Returns dictionary of all the styles currently defined in the UI
        """
        result = {}
        styles = self.Palette.GetStyles()
        for key in styles.Keys:
            style = Style(styles[key])
            result[key] = style
        return result

    def set_style(self, key: str, style: Style) -> None:
        """
        Set style data to the style in the UI
        """
        styles = self.Palette.GetStyles()
        if key in styles.Keys:
            style = Style(style)
            uiStyle = styles[key]
            uiStyle.Layer           = style.Layer
            uiStyle.Color           = style.Color
            uiStyle.Pen             = style.Pen
            uiStyle.Stroke          = style.Stroke
            uiStyle.ColorByLayer    = style.ColorByLayer
            uiStyle.PenByLayer      = style.PenByLayer
            uiStyle.StrokeByLayer   = style.StrokeByLayer

    def add_style(self, key: str) -> None:
        self.Palette.AddStyle(key)

    def clear_styles(self) -> None:
        self.Palette.ClearStyles()

    def get_input_options(self) -> InputOptions:
        """
        Get all the inputs defined in the UI (except styles)
        """
        optionsDict = dict()
        optionsDict["DropPoint"] = self.Palette.InputOptions.DropPointInt
        optionsDict["ChainingPoint"] = self.Palette.InputOptions.ChainingPointInt
        optionsDict["Catalogue"]     = self.Palette.InputOptions.Catalogue
        optionsDict["LoDThresholds"] = list(self.Palette.LevelOfDetailPicker.ThresholdList)
        
        # options = InputOptions()
        # options.DropPoint     = self.Palette.InputOptions.DropPointInt
        # options.ChainingPoint = self.Palette.InputOptions.ChainingPointInt
        # options.Catalogue     = self.Palette.InputOptions.Catalogue
        # options.LoDThresholds = list(self.Palette.LevelOfDetailPicker.ThresholdList)

        attributes = []
        for attribute in self.Palette.AttributeInput.AttributeList:
            id = int(attribute.Item1)
            value = str(attribute.Item2)
            attributeType = str(attribute.Item3)
            userDefined = bool(attribute.Item4)
            attributes.append((id, value, attributeType, userDefined))
        #attributes.append((289, self.Settings.LoadedSettings.SelectedSection, "String", False))
        attributes.append((289, PaletteWrapper.GetCountry(), "String", False))
        #options.Attributes = attributes
        optionsDict["Attributes"] = attributes
        return InputOptions(optionsDict)

    def set_input_options(self, options: InputOptions) -> None:
        """
        Sets input options in the UI
        """
        self.Palette.InputOptions.DropPointInt = options.DropPoint
        self.Palette.InputOptions.ChainingPointInt = options.ChainingPoint
        self.Palette.InputOptions.Catalogue = options.Catalogue
        
        lodList = Generic.List[System.Int32]()
        for threshold in options.LoDThresholds:
            lodList.Add(threshold)

        self.Palette.LevelOfDetailPicker.ThresholdList = lodList

        attributes = Generic.List[System.ValueTuple[System.Int32, System.String, System.String, System.Boolean]]()
        for id, value, attributeType, userDefined in options.Attributes:
            id = System.Int32(id)
            value = System.String(value)
            attributeType = System.String(attributeType)
            userDefined = System.Boolean(userDefined)
            attributes.Add(System.ValueTuple[System.Int32, System.String, System.String, System.Boolean](id, value, attributeType, userDefined))
        self.Palette.AttributeInput.AttributeList = attributes

    def get_product(self):
        return self.Palette.CurrentProduct

    ###################
    # Default setters #
    ###################

    def __try_set_default_section(self) -> None:
        """
        In the case the section is not selected, 
        this function mapps the language to a section and sets it
        French and Russian are still not enabled
        """
        if self.Settings.LoadedSettings.SelectedSection == "":
            currentLanguage = Settings.AllplanLocalisationService.Language()
            
            self.Settings.LoadedSettings.SelectedSection = get_section(currentLanguage)

    def __try_set_default_culture(self) -> None:
        """
        In the case the culture is not selected, 
        this function mapps the language to a culture and sets it
        """
        if self.Settings.LoadedSettings.SelectedCulture == "":
            currentLanguage = Settings.AllplanLocalisationService.Language()
            
            self.Settings.LoadedSettings.SelectedCulture = get_culture(currentLanguage)

    def __try_set_catalogue_selection(self) -> None:
        """
        Tries to get the precast catalog references for point fixtures
        If the references can be obtained, it sorts them, and adds them to the combobox in the UI
        """
        if int(Settings.AllplanVersion.MainReleaseName()) > 2022:
            self.Palette.InputOptions.CatalogueSelection.Clear()
            catalogueReferences = [str(x) for x in PaletteWrapper.GetPrecastCatalog(2)]
            catalogueReferences.sort()
            catalogueReferences.insert(0, "")
            for reference in catalogueReferences:
                self.Palette.InputOptions.CatalogueSelection.Add(reference)

            if self.Palette.InputOptions.Catalogue is None or len(self.Palette.InputOptions.Catalogue) == 0:
                self.Palette.InputOptions.Catalogue = catalogueReferences[0]
        
    ############
    # UserData #
    ############

    def upload_data(self, action: str, count: Optional[int] = None) -> None:
        """
        Uploads data about the project, inputs, version, action, count and total number of objects in the project
        """
        try:
            data = {
                "ProjectData": self.__get_attribute_data(),
                "Inputs": self.__get_plugin_data(),
                "Version": self.__get_versions(),
                "Action": action,
                "CurrentTotal": self.__get_total_product_count()
            }
            if count is not None:
                data["Count"] = count
            blob = json.dumps(data)
            self.SchoeckUI.Globals.UploadBlob(blob)
        except:
            print(f"Blob upload failed" )

    def __get_attribute_data(self) -> Dict[str, str]:
        """
        Gets project attribute data
        The data is loaded only once and cached as it cannot be modified while using the plugin
        """
        if self.__attributeData is None:
            attributes = BaseElements.ProjectAttributeService.GetAttributesFromCurrentProject()
            attributes = {k: v for k, v in attributes}

            data = {
                "Projektstatus":                        attributes.get(15  , ""),
                "Projektname":                          attributes.get(405 , ""),
                "Programmversion":                      attributes.get(449 , ""),
                "Bauvorhaben":                          attributes.get(826 , ""),
                "Bauvorhaben Anschrift":                attributes.get(921 , ""),
                "Bauvorhaben Straße":                   attributes.get(922 , ""),
                "Bauvorhaben PLZ/Ort":                  attributes.get(923 , ""),
                "Projektnummer":                        attributes.get(936 , ""),
                "Projekt":                              attributes.get(985 , ""),
                "Projektbeschreibung":                  attributes.get(991 , ""),
                "Projektname (FTW - Benutzereingabe)":  attributes.get(1023, ""),
                "Bauvorhaben Allgemein":                attributes.get(1091, ""),
                "Bauvorhaben Name":                     attributes.get(1093, ""),
                "Bauvorhaben Adresse":                  attributes.get(1094, ""),
                "Bauvorhaben Homepage":                 attributes.get(1388, ""),
                }
            self.__attributeData = data

        return self.__attributeData

    def __get_plugin_data(self) -> dict:
        """
        Get plugin data
        Current ProductId, ArticleNumber, Section and Culture
        as well as styles and input data
        """
        def serialize_data(data):
            attributes = [x for x in dir(data) if not x.startswith("__")]
            result = {}
            for attribute in attributes:
                # Ignore functions
                attr = getattr(data, attribute)
                if hasattr(attr, "__call__"):
                    continue
                result[attribute] = attr
            return result

        styles = self.get_styles()
        inputOptions = self.get_input_options()

        styleData = {k: serialize_data(styles[k]) for k in styles}
        inputOptionsData = serialize_data(inputOptions)
        
        data = self.Palette.CurrentProduct.Data
        productData = {
            "ProductId": data.product.parameters[2].value,
            "ArticleNumber": self.SchoeckUI.Globals.ProductUtil.GetAttributeValue(data, "ARTICLENO"),
            "Section": self.Settings.LoadedSettings.SelectedSection,
            "Culture": self.Settings.LoadedSettings.SelectedCulture,
            }

        return { "Styles": styleData, "Options": inputOptionsData, "Product": productData }

    def __get_versions(self) -> dict:
        """
        Gets version of Allplan and of the plugin dlls
        """
        allplan = Settings.AllplanVersion.Version()
        plugin = self.SchoeckUI.Globals.Version
        return { "AllplanVersion": allplan, "PluginVersion": plugin }

    def __get_total_product_count(self) -> int:
        """
        Counts all Schoeck product currently loaded in Allplan
        """
        tic = time.perf_counter()
        targetID =  538
        targetFilter = "Schöck Catalogue"
        total = 0
        for element in BaseElements.ElementsSelectService.SelectAllElements(self.Document):
            if element != ElementAdapter.PointFixture_TypeUUID:
                continue
            attributes = BaseElements.ElementsAttributeService.GetAttributes(element)

            for id, value in attributes:
                if id == targetID:
                    if value == targetFilter:
                        total += 1
                    else:
                        break
                if id > targetID:
                    break
                
        toc = time.perf_counter()
        print(f"Counted a total of {total} placed elements ({(toc - tic) * 1000:0.4f} ms)")
        return total