from tracemalloc import start
from xml.dom.minidom import Element
import NemAll_Python_Geometry as Geometry
import NemAll_Python_BaseElements as BaseElements
import NemAll_Python_IFW_Input as Input
import NemAll_Python_IFW_ElementAdapter as ElementAdapter

from HandleProperties import HandleProperties
from HandleDirection import HandleDirection

from typing import List, Literal
from enum import IntEnum

from .SchoeckUIBase import SchoeckUIBase

from ..Utilities.ProductHelper import is_cuttable
from ..Utilities.MouseMessageHelper import MouseMessageType

class GeometryMode(IntEnum):
    WHOLE = 0
    FIRST_HALF = 1
    SECOND_HALF = 2


class SchoeckModifyInteractor(SchoeckUIBase):
    """
    Definition of class UsingClrInteractor
    build_ele_list, build_ele_composite, control_props_list, modify_uuid_list
    """

    def __init__(self, *args, **kwargs):
        """
        Initialization of class UsingClrInteractor
        Args:
            coord_input:        coordinate input
            pyp_path:           path of the pyp file
            str_table_service:  string table service
        """
        super().__init__(*args, **kwargs)

        self.Palette.Selector.SetInteractorMode("MODIFY")
        self._is_loading = True

        self.Palette.ProductSet += lambda: self.product_set_handler()        
        self.InitialHash = self.BuildingElement.get_hash()

        self.ModelElements = []
        self.PreviewModelElements = []
        self.ActiveElement = ElementAdapter.BaseElementAdapter.FromNOIGUID(self.ModifyList[0], self.Document)
        placement: Geometry.Matrix3D = BaseElements.PythonPartService.GetPlacementMatrix(self.ActiveElement)[1]
        self.PlacementPosition: Geometry.Point3D = Geometry.Point3D() + placement.GetTranslationVector()
        self.FullPlacementPosition: Geometry.Point3D = Geometry.Point3D(self.PlacementPosition)
        self.Direction = Geometry.Vector3D(*self.get_parameter("Direction"))
        self.Direction.Normalize()

        cultureID = self.get_parameter("CultureID")
        sectionID = self.get_parameter("SectionID")
        self.Palette.SettingsView.LocalizationLocked = True
        self.Settings.LoadedSettings.SelectedCulture = cultureID
        self.Settings.LoadedSettings.SelectedSection = sectionID
        productID = self.get_parameter("ProductID")

        self.Palette.SetTab("Style")
        self.Palette.Selector.SetSearchEnabled(True)
        self.Palette.Selector.Navigate(productID, True)
        self.Palette.ApplyVisible = True
        self.ApplyHandler = lambda: self.apply_handler()
        self.Palette.Apply += self.ApplyHandler
        self.CoordinateInput.InitValueInput(Input.InputStringConvert(self.Localization.CurrentLanguage["Interactor_Modify"]), Input.ValueInputControlData())
    
        self.Mirror = bool(self.BuildingElement.Mirror.value)
        self.RedrawLambda = lambda s, e: self.draw_preview(1)
        self.Palette.InputChanged += self.RedrawLambda
        # Handles
        length = self.get_parameter("Length")
        self.GeometryMode = GeometryMode.WHOLE
        if abs(length) - 0.5 < 0.1:
            self.GeometryMode = GeometryMode.FIRST_HALF if length > 0 else GeometryMode.SECOND_HALF

        self.HandleService: Input.HandleService = Input.HandleService()
        self.HandleProps: List[HandleProperties] = []

    def get_palette_title(self) -> str:
        return self.Localization.CurrentLanguage["Modify_Title"]

    ########################
    # Interactor functions #
    ########################
    def on_cancel_function(self) -> bool:
        """
        Check for input function cancel in case of ESC
        Returns:
            True/False for success.
        """
        if self.InErrorState or self.InitialHash == self.BuildingElement.get_hash():
            # If the element was not modified, just restore the old state
            BaseElements.CreateElements(self.Document, Geometry.Matrix3D(), [], self.ModifyList, None)
        else:
            self.Preview = False
            self.create()
            BaseElements.CreateElements(self.Document, Geometry.Matrix3D(), self.ModelElements, self.ModifyList, None)
            self.upload_data("Modify")

        self.HandleService.RemoveHandles()
        self.Palette.Apply -= self.ApplyHandler
        self.Palette.InputChanged -= self.RedrawLambda
        return super().on_cancel_function()

    def on_mouse_leave(self) -> None:
        """
        Handles the mouse leave event
        """
        if not self.InErrorState:
            self.draw_preview()

    def process_mouse_msg(self, MouseMessage, Point, MessageInfo) -> bool:
        """
        Process the mouse message event
        Args:
            mouse_msg:  the mouse message.
            pnt:        the input point in view coordinates
            msg_info:   additional message info.
        Returns:
            True/False for success.
        """
        if not self.InErrorState:
                    
            index, matrix = self.HandleService.SelectHandle(Point, self.CoordinateInput.GetViewWorldProjection())
            handleModified = False
            if MouseMessage == MouseMessageType.MOUSE_DOWN:
                # If mouse down select the handle
                self.SelectedHandle = index
                self.lastSelectedHandle = self.SelectedHandle
                # If no handle was selected -> the user clicked outside -> finish selection
                if is_cuttable(self.get_product()):                
                    if index >= 0:
                        handleModified = True
                        if self.GeometryMode == GeometryMode.WHOLE:
                            if index == 0:
                                self.GeometryMode = GeometryMode.SECOND_HALF
                            elif index == 1:
                                self.GeometryMode = GeometryMode.FIRST_HALF
                        elif self.GeometryMode == GeometryMode.FIRST_HALF:
                            if index == 0:
                                self.GeometryMode = GeometryMode.SECOND_HALF
                            elif index == 1:
                                self.GeometryMode = GeometryMode.WHOLE 
                        elif self.GeometryMode == GeometryMode.SECOND_HALF:
                            if index == 0:
                                self.GeometryMode = GeometryMode.WHOLE
                            elif index == 1:
                                self.GeometryMode = GeometryMode.FIRST_HALF
                        
                        if self.GeometryMode == GeometryMode.SECOND_HALF:
                            self.PlacementPosition = self.FullPlacementPosition + self.Direction * (self.get_product().Length / 2)
                        else:
                            self.PlacementPosition = self.FullPlacementPosition
                        self.__create_handles()
                        
                if self.SelectedHandle != -1 and self.HandleProps[self.SelectedHandle].handle_id == "Mirror":
                    self.Mirror = not self.Mirror
                    self.BuildingElement.Mirror.value = self.Mirror
                    handleModified = True
                
            if MouseMessage == MouseMessageType.MOUSE_UP:
            # On mouse up, the handle is deselected
                self.SelectedHandle = -1
                    
            self.draw_preview(handleModified)
        return True

    def draw_preview(self, Force=False) -> None:
        """
        Draw the preview
        Args:
            InputPoint:  Input point
        """    
        if Force == True:
            self.create(True)
        if not self.InErrorState and len(self.PreviewModelElements) > 0:
            BaseElements.DrawElementPreview(self.CoordinateInput.GetInputViewDocument(), Geometry.Matrix3D(), self.PreviewModelElements, True, None)
            self.HandleService.DrawHandles()

    ###########
    # Helpers #
    ###########
    def create(self, Preview = False):
        """
        Creates model elements that are used for the preview and creation
        """
        styles = self.get_styles()        
        inputOptions = self.get_input_options()
        length = 1 if self.GeometryMode == GeometryMode.WHOLE else 0.5
        self.Mirror = self.BuildingElement.Mirror.value
        self.PreviewModelElements = self.create_object(
            self.get_product(), 
            self.Direction, 
            self.PlacementPosition, 
            inputOptions, 
            styles, 
            length, 
            startWithCut = self.GeometryMode == GeometryMode.SECOND_HALF,
            mirror = self.Mirror,
            preview=Preview)

    def product_set_handler(self):
        """
        Handles ProductSet event
        """

        def set_styles():

            

            itemGroups = set()
            for key in self.Geometries:
                itemGroups.update(self.Geometries[key])
            itemGroups = sorted(itemGroups)

            self.clear_styles()

            for group in itemGroups:
                style = self.add_style(str(group))

            if styles is not None:
                uiStyles = self.get_styles()
                for style in styles:
                    if style in uiStyles:
                        self.set_style(style, styles[style])


        def set_parameters():
            self.set_parameter("ProductID", self.get_product().Code)
            self.set_parameter("SectionID", self.Settings.LoadedSettings.SelectedSection)
            self.set_parameter("CultureID", self.Settings.LoadedSettings.SelectedCulture)

        def set_inputoptions():
            
            if inputOptions:
                self.set_input_options(inputOptions)

        styles = self.get_parameter("Styles")
        inputOptions = self.get_parameter("InputOptions")
        if self._is_loading:


            self.update_geometries(self.get_product())

            # Set the styles
            set_inputoptions()
            set_styles()
            


            # Finally everything is loaded
            # Set is loading to false
            self._is_loading = False
            self.__create_handles()
            return




        if self.get_product() is None:
            # If the product gets set to None, clear geometries and styles
            self.Geometries = {}
            self.clear_styles()
        else:
            # Set the parameters in pyp file
            set_parameters()

            # Try update geometries
            self.update_geometries(self.get_product())

            # Set the styles and input options defined in Pyp        
            set_styles()
            set_inputoptions()
            
            
            # Create model elements
            self.create()
            if self.GeometryMode == GeometryMode.SECOND_HALF:
                self.FullPlacementPosition = self.FullPlacementPosition - (self.Direction * (self.get_product().Length / 2))
            self.__create_handles()
        self.draw_preview()

    def apply_handler(self):
        if not self.InErrorState:
            # Create, draw and set the flag to modified
            self.create()
            self.ModelElements = self.PreviewModelElements

            # The chaining point identifier is ten times bigger in preview
            # therefore a seperation between the actual placed element,
            self.create(True)
            self.draw_preview(True)

    ###########
    # Handles #
    ###########            

    def __create_handles(self):
        productLength = self.get_product().Length
        end = productLength if self.GeometryMode == GeometryMode.WHOLE else productLength / 2            
        startPosition = Geometry.Point3D(self.PlacementPosition)
        directionVector = Geometry.Vector3D(self.Direction)
        directionVector.Normalize(end)
        endPosition = startPosition + directionVector
        if is_cuttable(self.get_product()):

            self.HandleProps.clear()
            handle = HandleProperties("Start", startPosition, Geometry.Point3D(), [], HandleDirection.VECTOR_DIR)
            handle.handle_type  = Input.ElementHandleType.HANDLE_SQUARE_RED
            self.HandleProps.append(handle)

            handle = HandleProperties("End", endPosition, Geometry.Point3D(), [], HandleDirection.VECTOR_DIR)
            handle.handle_type  = Input.ElementHandleType.HANDLE_SQUARE_RED
            self.HandleProps.append(handle)
            
        handle = HandleProperties("Mirror", (startPosition + endPosition) / 2, Geometry.Point3D(), [], HandleDirection.CLICK)
        handle.handle_type  = Input.ElementHandleType.HANDLE_ARROW
        handle.info_text    = "Mirror"
        rot_direction = Geometry.Vector2D(self.Direction.X, self.Direction.Y)        
        rot_angle = Geometry.Angle()
        rot_angle.SetDeg(90)
        handle.handle_angle = rot_angle + rot_direction.GetAngle() 
        self.HandleProps.append(handle)
    
        self.HandleService.RemoveHandles()
        self.HandleService.AddHandles(self.Document, self.HandleProps, Geometry.Matrix3D(), None)

