# Compiled Allplan Packages
from tracemalloc import start
from turtle import width
import NemAll_Python_Geometry as Geometry
import NemAll_Python_BaseElements as BaseElements
import NemAll_Python_BasisElements as BasisElements
import NemAll_Python_IFW_Input as Input
import NemAll_Python_IFW_ElementAdapter as ElementAdapter
import NemAll_Python_AllplanSettings as Settings
import NemAll_Python_Precast as Precast
import NemAll_Python_Utility as AllplanUtil

# Allplan script packages from PythonPartsFramework/GeneralScripts
import BuildingElement
import StringTableService
import BuildingElementComposite
import ControlProperties
import PythonPart
from BuildingElementPaletteService import BuildingElementPaletteService

# Python scripts
import os
import base64
import io
import pickle
import imp
import threading
import hashlib
import math
import json

from typing import List, Dict, Tuple, Optional

# Project packages

from ..Style import Style
from ..InputOptions import InputOptions

from ..Utilities.ProductHelper import is_upright, is_cuttable, get_length

# clr and related packages
import clr

class SchoeckInteractorBase:
    """
    Definition of class SchoeckInteractorBase
    It is the base function for the interactor and defines all the interaction between the interactor and the configuration
    It also defines some functions that are of use in every other inherited class
    Sets Allplan interaction classes (CoordinateInput, DocumentAdapter ...)
    
    """
    def __init__(self, 
                 coordinateInput: Input.CoordinateInput, 
                 pypPath: str, 
                 stringTable: StringTableService,
                 buildingElements: List[BuildingElement.BuildingElement],
                 buildingElementComposite: BuildingElementComposite.BuildingElementComposite,
                 controlProps: ControlProperties.ControlProperties,
                 modifyUUIDList: List) -> None:
        # Store arguments
        self.CoordinateInput: Input.CoordinateInput     = coordinateInput
        self.Document: ElementAdapter.DocumentAdapter   = coordinateInput.GetInputViewDocument()
        self.BuildingElement: BuildingElement.BuildingElement = buildingElements[0]
        self.ModifyList = modifyUUIDList
        self.PypPath: str = pypPath

        self.__currentProductCode = ""
        self.Geometries = {}
        self.CutGeometries = [{}, {}]
        self.InErrorState = False

    ############
    # Handlers #
    ############
    def enter_error_state(self, exception):
        if not self.InErrorState:
            self.InErrorState = True

    ######################
    # Abstract functions #
    ######################

    def draw_preview(self, Input: Optional[Geometry.Point3D]) -> None:
        """
        Function has to be overwritten by classes that inherit this
        """
        raise NotImplementedError("draw_preview not implemented!")

    ########################
    # Interactor functions #
    ########################
    def modify_element_property(self, page, name, value) -> None:
        """
        This function has to be defined by an interactor
        Useless for our purposes, as it has to do with the standard AllplanPalette which we are not using
        """
        pass

    def on_cancel_function(self) -> bool:
        """
        Check for input function cancel in case of ESC
        Returns: True/False for success.
        """
        return True

    def on_preview_draw(self) -> None:
        """
        Handles the preview draw event
        """
        self.draw_preview(self.CoordinateInput.GetCurrentPoint().GetPoint())

    def on_mouse_leave(self) -> None:
        """
        Handles the mouse leave event
        Useful when handling Static geometries
        """
        pass

    #########################
    # Pyp context functions #
    #########################

    def set_parameter(self, parameter: str, value: object) -> None:
        """
        Encodes and writes the value into the building element
        Throws KeyError if the building element does not contain the parameter
        """
        
        def encode_object(obj):  
            stream = io.StringIO()                    
            json.dump(obj.__dict__, stream) 
            stream.seek(0)
            b = stream.read()   
            stream.close()    
            sample_string_bytes = b.encode("utf-16")
            base64_bytes = base64.b64encode(sample_string_bytes)
            base64_string = base64_bytes.decode("utf-16")

            return base64_string
        
        if hasattr(self.BuildingElement, parameter):
            if parameter == "InputOptions":
                encoded = encode_object(value)
                getattr(self.BuildingElement, parameter).value = encoded
                return
            if parameter == "Styles":
                result_dict = dict()                 
                for key, v in value.items(): 
                    stream = io.StringIO()                  
                    json.dump(v.__dict__, stream) 
                    stream.seek(0)
                    b = stream.read() 
                    stream.close()     
                    result_dict[key] = b  
                stream = io.StringIO()   
                json.dump(result_dict, stream) 
                stream.seek(0)
                b = stream.read() 
                stream.close() 
                sample_string_bytes = b.encode("utf-16")
                base64_bytes = base64.b64encode(sample_string_bytes)
                base64_string = base64_bytes.decode("utf-16")
                getattr(self.BuildingElement, parameter).value = base64_string
                return
            #encoded = encode_object(value, parameter)
            getattr(self.BuildingElement, parameter).value = value
            return
        raise RuntimeError(f"Key {parameter} does not exist")

    def get_parameter(self, parameter: str) -> object:
        """
        Fetches the parameter from the building element, and decodes it
        Throws KeyError if the building element does not contain the parameter
        """
        def decode_pickle_object(obj, parameter):
            for _ in range(5):
                try:
                    decoded = base64.b64decode(obj)
                    stream = io.BytesIO()
                    stream.write(decoded)
                    stream.seek(0)
                    return pickle.load(stream)
                except:
                    continue
            if parameter == "InputOptions" or parameter == "Styles":
                AllplanUtil.ShowMessageBox("Input Options loading failed.\nThe Default Options will be loaded.", 0)
                return 
            
        def decode_input_options(obj):
            base64_bytes = obj.encode("utf-16")  
            sample_string_bytes = base64.b64decode(base64_bytes)
            sample_string = sample_string_bytes.decode("utf-16")
            stream = io.StringIO()
            stream.write(sample_string)
            stream.seek(0)
            return InputOptions(json.load(stream))
        
        def decode_styles(obj):
            base64_bytes = obj.encode("utf-16")  
            sample_string_bytes = base64.b64decode(base64_bytes)
            sample_string = sample_string_bytes.decode("utf-16")
            stream = io.StringIO()
            stream.write(sample_string)
            stream.seek(0)
            styles_dict_strem = json.load(stream)
            styles_dict = dict()
            for key, value in styles_dict_strem.items():
                stream = io.StringIO()
                stream.write(value)
                stream.seek(0)
                styles_dict[key] = Style(json.load(stream))
            stream.close() 
            return styles_dict
        
        def decode_direction(obj):
            directionString = str(obj)
            directionString = directionString.replace('(', '')
            directionString = directionString.replace(')', '')
            x, y, z = directionString.split(",")
            return (float(x), float(y), float(z))
        
        def decode_object(obj):
            if obj is None or len(obj) == 0:
                return None
            if parameter == "InputOptions":
                return decode_input_options(obj)
            if parameter == "Styles":
                return decode_styles(obj)
            if parameter == "Direction":
                return decode_direction(obj)
            if parameter == "Length":
                return float(obj)
            return obj
        
        if hasattr(self.BuildingElement, parameter):
            obj = getattr(self.BuildingElement, parameter).value
            version = getattr(self.BuildingElement, "Version").value
            if version != 'new':
                return decode_pickle_object(obj, parameter)
            return decode_object(obj)
        raise KeyError(f"Key {parameter} does not exist")

    #####################
    # Utility functions #
    #####################

    def get_dimensions(self, product):
        height = product.Height
        width = product.Width
        length = product.Length
        
        if not self.Geometries:
            return length, width, height
        
        minLod = min(self.Geometries)
        for color in self.Geometries[minLod]:
            for element in self.Geometries[minLod][color]:
                err, vertices = element.GetVertices()
                for v in vertices:
                    x,y,z = v.Values()
                    if (0, 0, 0) == (abs(round(x)), abs(round(y)), abs(round(z))):
                        length = max(p.X for p in vertices)
                        width = max(p.Y for p in vertices)
                        height = max(p.Z for p in vertices)
                        if height > 0:
                            return length, width, height   
                                        
        return length, width, height

    def get_local_transform(self, inputOptions, product) -> Geometry.Matrix3D:
        """
        Converts alignment and product dimensions to a local transformation matrix
        Positions (from AllplanRefPointButton):
            TopLeft = 1,
            TopCenter = 2,
            TopRight = 3,
            BottomLeft = 7,
            BottomCenter = 8,
            BottomRight = 9
        """

        _, width, height = self.get_dimensions(product)
        # Center of the object
        pos_z = height / 2
        pos_y = width / 2
        dz = height / 2
        dy = width / 2

        # 0 - Left | 1 - Center | 2 - Right 
        # 0 - Top | 1 - Center | 2 - Bottom 
        position_vertical = (inputOptions.DropPoint - 1) // 3
        position_horizontal = (inputOptions.DropPoint - 1) % 3

        # If not center
        if position_vertical != 1:
            pos_z += dz if position_vertical == 0 else -dz
        if position_horizontal != 1:
            pos_y += dy if position_horizontal == 0 else -dy
        
        matrix = Geometry.Matrix3D()
        matrix.Translate(Geometry.Vector3D(0, -pos_y, -pos_z))
        return matrix

    def update_geometries(self, product, cutRatio: float = 0.5):
        """
        Checks if the current geometries are up to date
        If they have to be updated:
            Fetches geometry scripts from the product for different levels of detail
            Parses every script and extracts geometries
        """
        
        if product.Code == self.__currentProductCode:
            return
        
        self.Geometries = {}
        self.CutGeometries = [{}, {}]
        createCut = is_cuttable(product)
        cutLength = get_length(product) * cutRatio
        cutPlane = Geometry.Plane3D(Geometry.Point3D(cutLength, 0, 0), Geometry.Vector3D(1, 0, 0))
        secondCutTranslation = Geometry.Matrix3D()
        secondCutTranslation.Translate(Geometry.Vector3D(-cutLength, 0, 0))
            
        for key in product.Geometries.Keys:
            script = product.Geometries[key]
            pythonPartModule = imp.new_module("pythonPartModule")
            exec(script, pythonPartModule.__dict__)
            pythonPart, handles = pythonPartModule.create_element(self.BuildingElement, self.Document)
            elements = pythonPart[0].GetSlideList()[1].GetObjectList()
            mapping = {}
            firstCutMapping = {}
            secondCutMapping = {}
            for element in elements:
                commonProperties = element.GetCommonProperties()
                color = commonProperties.Color
                if color not in mapping:
                    mapping[color] = []
                    firstCutMapping[color] = []
                    secondCutMapping[color] = []
                
                geometry = element.GetGeometryObject()
                mapping[color].append(geometry)
                if createCut:
                    if isinstance(geometry, Geometry.BRep3D):
                        success, up, bellow = Geometry.CutBrepWithPlane(geometry, cutPlane)
                    if isinstance(geometry, Geometry.Polyhedron3D):
                        success, up, bellow = Geometry.CutPolyhedronWithPlane(geometry, cutPlane)
                    
                    if bellow.IsValid():
                        firstCutMapping[color].append(bellow)
                    if up.IsValid():
                        up = Geometry.Transform(up, secondCutTranslation)
                        secondCutMapping[color].append(up)
            self.Geometries[key] = mapping
            self.CutGeometries[0][key] = firstCutMapping
            self.CutGeometries[1][key] = secondCutMapping
            
        product.Code = self.__currentProductCode

    def __get_hash(self, prefix: str) -> str:
        buildEleHash = self.BuildingElement.get_hash()
        hashValue = prefix + buildEleHash
        return hashlib.sha224(hashValue.encode('utf-8')).hexdigest()

    def __cast_attributes(self, attributes: List[Tuple[int, str, str, bool]]) -> List[BaseElements.Attribute]:
        """
        Creates list of attributes from product
        """
        res = []
        ids = set()
        for id, value, attributeType, userDefined in attributes:
            if id in ids:
                continue
            try:
                if attributeType == "Integer":
                    res.append(BaseElements.AttributeInteger(id, int(value)))
                    ids.add(id)
                elif attributeType == "Double":
                    res.append(BaseElements.AttributeDouble(id, float(value)))
                    ids.add(id)
                elif attributeType == "String":
                    res.append(BaseElements.AttributeString(id, value))
                    ids.add(id)
                elif attributeType == "Enum":
                    valueId = BaseElements.AttributeService.GetEnumIDFromValueString(id, value)
                    res.append(BaseElements.AttributeEnum(id, valueId))
                    ids.add(id)
                elif attributeType == "Date":
                    tokens = [int(x) for x in value.split('/')]
                    res.append(BaseElements.AttributeDate(id, tokens[0], tokens[1], tokens[2]))
                    ids.add(id)
            except ValueError:
                print(f"ValueError encountered while parsing attribute with {id=} and {value=}")
        return res

    def __cast_range(self, thresholds: List[int]) -> Dict[int, Tuple[int, int]]:
        """
        Transforms LoD thresholds to a dictionary of ranges
        """
        return { 0: (thresholds[3], thresholds[4]),
                 1: (thresholds[2], thresholds[3]),
                 2: (thresholds[1], thresholds[2]),
                 3: (thresholds[0], thresholds[1])}

    def __create_measure_points(self, product, localTransform: Geometry.Matrix3D, isCut = False) -> List[BasisElements.Symbol3DElement]:
        symbolProperties = BasisElements.Symbol3DProperties()
        symbolProperties.SymbolID                = 1
        symbolProperties.Height                  = 1
        symbolProperties.Width                   = 1
        symbolProperties.IsScaleDependent        = False

        commonProperties = BaseElements.CommonProperties()
        commonProperties.HelpConstruction = True

        length, width, height = self.get_dimensions(product)
        length = length / 2 if isCut else length
        endPoint = Geometry.Point3D(length, width, height)
        endPoint = Geometry.Transform(endPoint, localTransform)

        startPoint = Geometry.Point3D(0, 0, 0)
        startPoint = Geometry.Transform(startPoint, localTransform)
        return [
            BasisElements.Symbol3DElement(commonProperties, symbolProperties, startPoint),
            BasisElements.Symbol3DElement(commonProperties, symbolProperties, endPoint),
        ]

    def __create_fixture_slide(self, 
            modelElements: List[BasisElements.AllplanElement], 
            startScale: int, 
            endScale: int, 
            fixtureViewType: Precast.FixtureSlideViewType = Precast.FixtureSlideViewType.e3D_VIEW, 
            visibility2D: bool = True, 
            visibility3D: bool = True) -> Precast.FixtureSlideElement:
        slideProperties = Precast.FixtureSlideProperties()

        slideProperties.Type = Precast.FixtureSlideType.eGeometry 
        slideProperties.ViewType = fixtureViewType
        slideProperties.VisibilityGeo3D = visibility2D
        slideProperties.VisibilityGeo2D = visibility3D
        slideProperties.VisibilityLayerA = True
        slideProperties.VisibilityLayerB = True
        slideProperties.VisibilityLayerC = True
        slideProperties.StartScaleRange = startScale
        slideProperties.EndScaleRange = endScale

        return Precast.FixtureSlideElement(slideProperties, modelElements)

    def __create_chaining_point(self, 
            inputOptions, 
            product, 
            localTransform: Geometry.Matrix3D, 
            isCut: bool = False,
            preview: bool = False) -> BasisElements.Symbol3DElement:
        # Positions (from AllplanRefPointButton):
        #     TopLeft     = 1
        #     TopCenter   = 2
        #     TopRight    = 3
        #     BottomLeft  = 7
        #     BottomCenter= 8
        #     BottomRight = 9
        
        chaining_point_offset = 50
        length, width, height = self.get_dimensions(product)

        length = length / 2 if isCut else length
        # Center of the object
        pos_x = length / 2
        pos_z = height / 2
        pos_y = width / 2

        # If the object is not upright, eg. Sconnex then the chaining points are above and bellow, and not left and right
        if is_upright(product):
            dz = height / 2
            dy = width / 2 + chaining_point_offset
        else:
            dz = height / 2 + chaining_point_offset
            dy = width / 2

        # 0 - Left | 1 - Center | 2 - Right 
        # 0 - Top | 1 - Center | 2 - Bottom 
        position_vertical = (inputOptions.ChainingPoint - 1) // 3
        position_horizontal = (inputOptions.ChainingPoint - 1) % 3

        # If not center
        if position_vertical != 1:
            pos_z += dz if position_vertical == 0 else -dz
        if position_horizontal != 1:
            pos_y += dy if position_horizontal == 0 else -dy

        # Define properties
        symbolProperties = BasisElements.Symbol3DProperties()
        symbolProperties.SymbolID                = 1
        symbolProperties.Height                  = 50 * 10 ** preview
        symbolProperties.Width                   = 50 * 10 ** preview
        symbolProperties.IsScaleDependent        = False

        # Render
        commonProperties = BaseElements.CommonProperties()
        commonProperties.HelpConstruction = True
        point = Geometry.Point3D(pos_x, pos_y, pos_z)
        point = Geometry.Transform(point, localTransform)
        return BasisElements.Symbol3DElement(commonProperties, symbolProperties, point)
    
    def __create_callback(self, 
            inputOptions, 
            localTransform: Geometry.Matrix3D, 
            mirrorTransform: Geometry.Matrix3D, 
            geometries: Dict[int, Dict[int, List[object]]], 
            styles: Dict[str, Style], 
            product, 
            isCut: bool = False,
            preview: bool = False) -> List[BasisElements.MacroSlideElement]:
        views = []
        rangesMin = min(inputOptions.LoDThresholds)
        rangesMax = max(inputOptions.LoDThresholds)
        minLod = min(geometries)
        callbackCommonProperties = BaseElements.CommonProperties()
        callbackCommonProperties.HelpConstruction = True
        modelElements = []
        
        commonPropertiesDictionary = { s: styles[s].CommonProperties() for s in styles}
        for groupKey in geometries[minLod]:
            for geometry in geometries[minLod][groupKey]:
                rotatededGeometry = Geometry.Transform(geometry, mirrorTransform)
                transformedGeometry = Geometry.Transform(rotatededGeometry, localTransform)
                
                # Get the group string which is the key for the style
                groupStr = str(groupKey)
                
                if groupStr in commonPropertiesDictionary:
                    commonProperties = commonPropertiesDictionary[groupStr]
                else:
                    commonProperties = callbackCommonProperties
                    
                modelElements.append(BasisElements.ModelElement3D(commonProperties, transformedGeometry))

        if preview:
            chainingPoint = self.__create_chaining_point(inputOptions, product, localTransform, isCut, preview)
            modelElements.append(chainingPoint)
            
        views.append(PythonPart.View2D3D(modelElements, rangesMin, rangesMax))
        return views

    def __create_fixture(self, 
            inputOptions, 
            localTransform: Geometry.Matrix3D, 
            mirrorTransform: Geometry.Matrix3D, 
            geometries: Dict[int, Dict[int, List[object]]], 
            styles: Dict[str, Style], 
            product, 
            lod: int=None,
            isCut: bool=False) -> Precast.FixtureElement:
        # Min and max LoD
        minLod = min(inputOptions.LoDThresholds)
        maxLod = max(inputOptions.LoDThresholds)

        # Chaining point
        chainingPoint = self.__create_chaining_point(inputOptions, product, localTransform, isCut)
        chainingPointSlide = self.__create_fixture_slide([chainingPoint], minLod, maxLod, Precast.FixtureSlideViewType.eCONNECTION_POINT)
        
        # Measure points
        measurePoints = self.__create_measure_points(product, localTransform, isCut)
        measurePointsSlide = self.__create_fixture_slide(measurePoints, minLod, maxLod, Precast.FixtureSlideViewType.eMEASURE_POINTS)

        # Init variables
        slides = [chainingPointSlide, measurePointsSlide]
        ranges = self.__cast_range(inputOptions.LoDThresholds)            
        commonPropertiesDictionary = { s: styles[s].CommonProperties() for s in styles}
            
        # Create geometries
        for key in geometries:
            modelElements = []
            # Define start end end scale for the slide
            start, end = ranges[key]
            # In the case start == end, skip slide
            if end - start <= 0:
                continue
            # In the case only one LoD has to be rendered
            if lod is not None:
                if lod == key:
                    start = min(inputOptions.LoDThresholds)
                    end = max(inputOptions.LoDThresholds)
                else:
                    continue
            # For every geometry in every geometry group
            for group in geometries[key]:
                for geometry in geometries[key][group]:
                    # Get the group string which is the key for the style
                    groupStr = str(group)

                    # Try get CommonProperties in this order: styles[key] > styles[default] > CommonProperties()
                    if groupStr in commonPropertiesDictionary:
                        commonProperties = commonPropertiesDictionary[groupStr]
                    elif "Default" in commonPropertiesDictionary:
                        commonProperties = commonPropertiesDictionary["Default"]
                    else:
                        commonProperties = BaseElements.CommonProperties()

                    # Transform and create ModelElement
                    rotatededGeometry = Geometry.Transform(geometry, mirrorTransform)
                    transformedGeometry = Geometry.Transform(rotatededGeometry, localTransform)
                    modelElements.append(BasisElements.ModelElement3D(commonProperties, transformedGeometry))
                
                    # This sets the Ifc type of every element to IfcMemeber, create mapping 
                    # attributeList = [BaseElements.AttributeString(684, "IfcMember")]
                    # attributes = BaseElements.Attributes([BaseElements.AttributeSet(attributeList)])
                    # modelElements[-1].SetAttributes(attributes)

            # Create fixture slide
            slides.append(self.__create_fixture_slide(modelElements, start, end))

        # Create fixture properties
        fixtureProps = Precast.FixtureProperties()
        fixtureProps.Type = Precast.MacroType.ePoint_Fixture
        fixtureProps.CatalogName = inputOptions.Catalogue
        fixtureProps.SubType = Precast.MacroSubType.eUseNoSpecialSubType
        fixtureProps.Name = "Schöck Isokorb"
        fixtureElement = Precast.FixtureElement(fixtureProps, slides)
        fixtureElement.SetHash(self.__get_hash("Fixture"))
        return fixtureElement

    def __create_fixture_placement(self, 
            fixture: Precast.FixtureElement, 
            attributeList: List[BaseElements.Attribute],
            placement: Geometry.Matrix3D = Geometry.Matrix3D()) -> Precast.FixturePlacementElement:
        # Create fixture placement properties
        fixturePlacementProps = Precast.FixturePlacementProperties()
        fixturePlacementProps.Name = "Schöck Isokorb"
        fixturePlacementProps.SubType = Precast.SubType.eUseSameSubType
        fixturePlacementProps.OutlineType = Precast.OutlineType.eBUILTIN_OUTLINE_TYPE_NO_AFFECT
        fixturePlacementProps.ConnectionToAIACatalog = True
        fixturePlacementProps.Matrix = placement

        # Add catalogue Attribute to the attributeList
        attributeList = [x for x in attributeList]
        # Cast attributes to be written in the element
        attributes = BaseElements.Attributes([BaseElements.AttributeSet(attributeList)])

        # Create fixture placement
        fixtureCommonProperties = BaseElements.CommonProperties()
        fixturePlacement = Precast.FixturePlacementElement(fixtureCommonProperties, fixturePlacementProps, fixture)
        fixturePlacement.SetAttributes(attributes)
        return fixturePlacement

    def __create_elements(self, 
            placement: Geometry.Matrix3D, 
            views: List[BasisElements.MacroSlideElement], 
            fixture: Precast.FixtureElement,
            attributeList: List[BaseElements.Attribute],
            fixtureOnly: bool = False,
            preview: bool = False) -> List:
        """
        Create elements that are going to be inserted
        Depending on the version of Allplan, we might want to create a callback PythonPart or just the fixture
        """
        if fixtureOnly and not preview:
            fixturePlacement = self.__create_fixture_placement(fixture, attributeList, placement)
            return [fixturePlacement]
        else:
            fixturePlacement = self.__create_fixture_placement(fixture, attributeList, Geometry.Matrix3D())
            pythonPart = PythonPart.PythonPart(name             = "Schöck Katalog PythonPart",
                                                parameter_list   = self.BuildingElement.get_params_list(),
                                                hash_value       = self.__get_hash("PythonPart"),
                                                python_file      = self.BuildingElement.pyp_file_name,
                                                views            = views,
                                                matrix           = placement,
                                                fixture_elements = [fixturePlacement])
            if hasattr(pythonPart, "leading_macro"):
                pythonPart.leading_macro(True)
            return pythonPart.create()

    def create_object(self, 
            product, 
            direction: Geometry.Vector3D, 
            position: Geometry.Point3D, 
            inputOptions, 
            styles: Dict[str, Style],
            count: float, 
            lod: Optional[int] = None,
            startWithCut: bool = False,
            mirror: bool = False,
            preview: bool = False
            ) -> List[BasisElements.AllplanElement]:
        # Get geometries
        
        self.BuildingElement.Version.value = "new"
        
        self.update_geometries(product)
        
        mirrorMatrix = Geometry.Matrix3D()
        cutMirrorMatrix = Geometry.Matrix3D()
        if mirror:
            length, width, height = self.get_dimensions(product)
            centerLine = Geometry.Line3D(Geometry.Point3D(length / 2, width / 2, 0), Geometry.Point3D(length / 2, width / 2, height))
            mirrorAngle = Geometry.Angle()
            mirrorAngle.SetDeg(180)
            mirrorMatrix.SetRotation(centerLine, mirrorAngle)
            centerLine = Geometry.Line3D(Geometry.Point3D(length * 0.25, width / 2, 0), Geometry.Point3D(length * 0.25, width / 2, height))
            cutMirrorMatrix.SetRotation(centerLine, mirrorAngle)
            
        # Get presentation variables
        localTransform = self.get_local_transform(inputOptions, product)
        attributes = self.__cast_attributes(inputOptions.Attributes)

        self.set_parameter("InputOptions", inputOptions)
        self.set_parameter("Styles", styles)
        self.set_parameter("Direction", (direction.X, direction.Y, direction.Z))
        self.set_parameter("Length", 1)
        self.BuildingElement.Mirror.value = mirror
        
        # if mirror:
        #     direction *= Geometry.Vector3D(-1,-1,-1)
        # Create PythonPart callback geometry
        views = self.__create_callback(inputOptions, localTransform, mirrorMatrix, self.Geometries, styles, product, False, preview)

        # Create fixture geometries and slides
        fixture = self.__create_fixture(inputOptions, localTransform, mirrorMatrix, self.Geometries, styles, product, lod)

        # Calculate rotation
        rotationZ = Geometry.Matrix3D()
        zAxis = Geometry.Line3D(Geometry.Point3D(), Geometry.Point3D(0, 0, 1))
        directionZ = Geometry.Vector2D(direction.X, direction.Y)
        angle = directionZ.GetAngle()
        rotationZ.Rotation(zAxis, angle)

        rotationY = Geometry.Matrix3D()
        yAxis = Geometry.Line3D(Geometry.Point3D(), Geometry.Point3D(0, -1, 0))
        directionY = Geometry.Vector2D(directionZ.GetLength(), direction.Z)
        angle = directionY.GetAngle()
        rotationY.Rotation(yAxis, angle)
        rotation = rotationY * rotationZ

        # Normalize direction
        lengthDirection = Geometry.Vector3D(direction)
        lengthDirection.Normalize(get_length(product))
        # Calculate start of the line
        rowStart = Geometry.Point3D(position)
        createCut = abs((count % 1) - 0.5) < 0.01
        if startWithCut and createCut:
            rowStart = rowStart + lengthDirection / 2

        # Check if only the fixture should be created
        fixtureOnly = int(Settings.AllplanVersion.MainReleaseName()) <= 2022

        result = []
        # Create whole elements
        for i in range(int(math.floor(count))):
            translation = Geometry.Matrix3D()
            startPosition = rowStart + (lengthDirection * i)
            translation.Translate(Geometry.Vector3D(startPosition))
            placement = rotation * translation

            result.extend(self.__create_elements(placement, views, fixture, attributes, fixtureOnly, preview))
        
        # Create cut elements
        if createCut:
            # Get values that depend on the position of the cut
            cutAttributes = list()
            for attribute in attributes:
                if attribute.Id == 1831:
                    cutAttributes.append(BaseElements.AttributeDouble(1831, attribute.Value / 2.))
                    continue
                cutAttributes.append(attribute)
            #attributes.append(BaseElements.AttributeDouble(1831, 0.5))
            if startWithCut:
                startPosition = position
                cutGeometries = self.CutGeometries[1]
                length = -0.5
            else:
                startPosition = position + (lengthDirection * int(math.floor(count)))
                cutGeometries = self.CutGeometries[0]
                length = 0.5
                
            # Calculate placement matrix
            translation = Geometry.Matrix3D()
            translation.Translate(Geometry.Vector3D(startPosition))
            placement = rotation * translation
            
            # Parameter Length has to be set before the fixture gets created as it is taken into account when calculating the hash
            self.set_parameter("Length", length)

            # Create PythonPart callback geometry
            views = self.__create_callback(inputOptions, localTransform, cutMirrorMatrix, cutGeometries, styles, product, True, preview)
            # Create fixture geometries and slides
            fixture = self.__create_fixture(inputOptions, localTransform, cutMirrorMatrix, cutGeometries, styles, product, lod, True)
            # Finally create elements
            result.extend(self.__create_elements(placement, views, fixture, cutAttributes, fixtureOnly))
        return result
