# Allplan imports
import NemAll_Python_Geometry as Geometry
import NemAll_Python_BasisElements as BasisElements
import NemAll_Python_IFW_Input as Input


from HandleDirection import HandleDirection
from HandleProperties import HandleProperties

# Python imports
import math
from typing import Dict, Literal, Optional, List

# Local imports

from ..Utilities.ProductHelper import is_cuttable
from ..Utilities.StateBehaviourType import StateBehaviourType

from .StateBehaviour import StateBehaviour
from .SchoeckUIBase import SchoeckUIBase
        

from .States.ProductSelectState import ProductSelectState
from .States.InputStartState import InputStartState
from .States.InputEndState import InputEndState
from .States.HandleState import HandleState

class SchoeckCreateInteractor(SchoeckUIBase):
    def __init__(self, match, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.HandleService: Input.HandleService = Input.HandleService()
        self.StartPoint: Geometry.Point3D = Geometry.Point3D()
        self.Direction: Geometry.Vector3D = Geometry.Vector3D(1, 0, 0)

        self.behaviours: Dict[StateBehaviourType, StateBehaviour] = {
            StateBehaviourType.PRODUCT_SELECT: ProductSelectState(self),
            StateBehaviourType.INPUT_START: InputStartState(self),
            StateBehaviourType.HANDLE: HandleState(self),
            StateBehaviourType.INPUT_END: InputEndState(self),
        }

        self.State: StateBehaviourType = StateBehaviourType.PRODUCT_SELECT
        self.CurrentBehaviour: Optional[StateBehaviour] = self.behaviours[self.State]
        self.ProductSetHandler = lambda: self.product_set_handler()
        self.Palette.ProductSet += self.ProductSetHandler
        self.Match: bool = match

        if self.Match:
            cultureID = self.get_parameter("CultureID")
            sectionID = self.get_parameter("SectionID")
            self.Palette.SettingsView.LocalizationLocked = True
            self.Settings.LoadedSettings.SelectedCulture = cultureID
            self.Settings.LoadedSettings.SelectedSection = sectionID
            productID = self.get_parameter("ProductID")

            self.Direction = Geometry.Vector3D(*self.get_parameter("Direction"))

            self.Palette.SetTab("Style")
            self.Palette.Selector.SetSearchEnabled(False)
            self.Palette.Selector.Navigate(productID, False)
            self.Palette.ApplyVisible = False
        else:
            self.change_state(StateBehaviourType.PRODUCT_SELECT)

            self.Palette.Selector.Navigate("Index")
            self.Palette.ApplyVisible = False

    def get_palette_title(self) -> str:
        return self.Localization.CurrentLanguage["Create_Title"]

    ########################
    # Interactor functions #
    ########################
    def on_cancel_function(self):
        if self.InErrorState or self.CurrentBehaviour.on_cancel_function():
            self.CurrentBehaviour.exit(StateBehaviourType.EXIT)
            self.Palette.ProductSet -= self.ProductSetHandler
            self.Palette.WriteSettings()
            self.SchoeckUI.Globals.Settings.SaveSettings()
            return super().on_cancel_function()
        return False

    def on_mouse_leave(self):
        """
        Handles the mouse leave event
        """
        if self.InErrorState:
            return
        self.CurrentBehaviour.on_mouse_leave()

    def draw_preview(self, InputPoint):
        if self.InErrorState:
            return
        self.CurrentBehaviour.draw_preview(InputPoint)

    def process_mouse_msg(self, MouseMessage, Point, MessageInfo):
        if self.InErrorState:
            return True
        return self.CurrentBehaviour.process_mouse_msg(MouseMessage, Point, MessageInfo)

    ###########
    # Helpers #
    ###########
    def change_state(self, nextState: StateBehaviourType) -> None:
        if not self.InErrorState:
            previousState = self.State
            self.State = nextState
            print(f"Changing state from {previousState} to {nextState}")
            if self.CurrentBehaviour:
                self.CurrentBehaviour.exit(nextState)
            self.CurrentBehaviour = self.behaviours[nextState]
            self.CurrentBehaviour.enter(previousState)

    def product_set_handler(self) -> None:
        """
        Handles the event when the product gets set to a value or None
        """
        if self.InErrorState:
            return

        if self.Palette.CurrentProduct == None:
            # If product gets set to None, go back to the PRODUCT_SELECT state
            self.change_state(StateBehaviourType.PRODUCT_SELECT)
        else:
            # In case the CurrentProduct gets set to a value,
            # write the parameters to the pyp file context
            self.set_parameter("ProductID", self.get_product().Code)
            self.set_parameter("SectionID", self.Settings.LoadedSettings.SelectedSection)
            self.set_parameter("CultureID", self.Settings.LoadedSettings.SelectedCulture)

            # Try update geometries
            self.update_geometries(self.get_product())

            # Get all the item groups of the geometries
            # Every item group has a different Color
            itemGroups = set()
            for key in self.Geometries:
                itemGroups.update(self.Geometries[key])
            itemGroups = sorted(itemGroups)

            # Clear the old styles
            self.clear_styles()
            #if len(itemGroups) == 1:
            #    # In the case there is only one item group, then the Default style is added
            #    self.add_style("Default")
            #else:
            # In the case there is several, then all are added, where the key is the color
            for group in itemGroups:
                style = self.add_style(str(group))

            # If in match mode, set the styles and input options 
            # of the matching object
            if self.Match:
                styles = self.get_parameter("Styles")
                if styles is not None:
                    uiStyles = self.get_styles()
                    for style in styles:
                        if style in uiStyles:
                            self.set_style(style, styles[style])
                inputOptions = self.get_parameter("InputOptions")
                if inputOptions:
                    self.set_input_options(inputOptions)
            # Go into INPUT_START mode
            self.change_state(StateBehaviourType.INPUT_START)

    def get_product_count(self, vector: Geometry.Vector3D) -> float:
        """
        Get the number of products that can be placed along a vector
        """
        lineLength = vector.GetLength()
        productLength = self.get_product().Length

        if is_cuttable(self.get_product()):
            # Products of length 1000 can be cut in half
            modulo = lineLength % (productLength / 2)
            productCount = (lineLength - modulo) / productLength
            increment = 0.5
        else:
            # Products of other lengths can only be placed whole
            productCount = math.floor(lineLength / productLength)
            increment = 1.

        # If the line is exactly a multiple of the length, dont add another object
        # If it is slightly over the multiple, add the increment
        if lineLength - productLength * productCount > 1:
            productCount += increment
        return productCount

    def create(self, start: Geometry.Point3D, direction: Geometry.Vector3D, lod: Optional[int] = None, startWithCut: bool = False, mirror: bool = False, preview: bool = False) -> List[BasisElements.ModelElement3D]:
        productCount = self.get_product_count(direction)
        styles = self.get_styles()
        inputOptions = self.get_input_options()
        mirror = self.BuildingElement.Mirror.value
        direction = Geometry.Vector3D(direction)
        direction.Normalize()
        return self.create_object(self.get_product(), direction, start, inputOptions, styles, productCount, lod, startWithCut, mirror, preview)